﻿-- AUTEUR(S) : Emmanuel Raoult
-- HISTORIQUE : 28/05/2015 - création du script pour import de la table de correspondance
-- RESUME : script d'import de l'Enquête Globale Déplacements 2014 de Montpellier (EGD 2014)
--		Cette enquête se décompose en une Enquête Déplacement Grand Territoire (EDGT)
--		et une Enquête Déplacement Villes Moyennes (EDVM).
--		Ce script normalise les zones fines des pôles générateurs, les identifiants des
--		ménages étudiants enquêté ainsi que la hiérarchie des modes urbains et inter urbains
-- A NOTER:	Attention à corriger la taille des variable utilisant échantillon (peut monter à
--		4 caractères dans EDVM 14)
--
-- SOMMAIRE :
-- 0. INTEGRATION DES TABLES GEOGRAPHIQUES ET DE CORRESPONDANCE D10-D30
-- I. INTEGRATION EDGT FACE A FACE								
-- II. INTEGRATION EDGT TELEPHONE								
-- III. INTEGRATION EDVM
-- IV. CONTROLE : Vérification des grands volumes									
--
---------------------------------------------------------------------------------------------------
--
---------------------------------------------------------------------------------------------------
-- PREREQUIS : INTEGRATION TABLE DE NORMALISATION						---
---------------------------------------------------------------------------------------------------
-- DROP TABLE metropole.egd_14_normalisation;
CREATE TABLE metropole.egd_14_normalisation (
	corresp_exacte varchar(3),
	table_emd varchar(24),
	variable_lib_edgt_faf varchar(140),
	variable_edgt_faf varchar(8),
	variable_edvm_edgt_tel varchar(8),
	variable_edvm_edgt_tel_lib varchar(140)
	);
-- puis import CSV 
---------------------------------------------------------------------------------------------------
--												---
-- 0. INTEGRATION DES TABLES GEOGRAPHIQUES ET DE CORRESPONDANCE D10-D30				---
--												---
---------------------------------------------------------------------------------------------------
-- 1. Import des tables de correspondance
-- 1.a. Création des tables de découpage interne et externe au périmètre
CREATE TABLE metropole.egd_14_decoupage_interne (
	sect_tir varchar(3),
	libsect_tir varchar(33),
	enquete varchar(6),
	libepci varchar(40),
	d10 integer,
	libd10 varchar(40),
	d30 integer,
	libd30 varchar(40)
	);
-- puis import CSV 
ALTER TABLE metropole.egd_14_decoupage_interne RENAME COLUMN sect_tir TO tira;
ALTER TABLE metropole.egd_14_decoupage_interne RENAME COLUMN libsect_tir TO tira_lib;
--
CREATE TABLE metropole.egd_14_decoupage_externe (
	zone_fine varchar(7),
	libzone_fine varchar(40),
	libsect_tir varchar(24),
	d10 integer,
	libd10 varchar(40),
	d30 integer,
	libd30 varchar(40)
	);
-- puis import CSV 
-- 1.b. Normalisation
-- i. Création des pseudo secteurs de tirage dans le découpage interne
ALTER TABLE metropole.egd_14_decoupage_interne ADD COLUMN tira_prefix_zone_fine_uid varchar(3);
UPDATE metropole.egd_14_decoupage_interne SET tira_prefix_zone_fine_uid = tira;
-- ii. Renommage des colonnes de libellés
ALTER TABLE metropole.egd_14_decoupage_interne RENAME libd10 TO d10_lib;
ALTER TABLE metropole.egd_14_decoupage_interne RENAME libd30 TO d30_lib;
-- iii. Création des pseudo secteurs de tirage dans le découpage externe
ALTER TABLE metropole.egd_14_decoupage_externe ADD COLUMN tira_prefix_zone_fine_uid varchar(3);
UPDATE metropole.egd_14_decoupage_externe SET tira_prefix_zone_fine_uid = substr(zone_fine,1, 3);
-- iv. Création de la table de correspondance complète
CREATE TABLE metropole.egd_14_decoupages_d10_d30 AS (
	SELECT tira_prefix_zone_fine_uid, d10, d10_lib, d30, d30_lib
	FROM metropole.egd_14_decoupage_interne
	UNION
	SELECT tira_prefix_zone_fine_uid, d10, libd10 AS d10_lib, d30, libd30 AS d30_lib
	FROM metropole.egd_14_decoupage_externe
	GROUP BY tira_prefix_zone_fine_uid, d10, libd10, d30, libd30
	)
cOMMENT ON COLUMN metropole.emd_14_decoupage_externe.tira_prefix_zone_fine_uid IS 'Code de zone de déplacement normalisé <pseudo secteur de tirage externe> (3 chiffres) + <pseudo zone fine> (3 chiffres) '
--
---------------------------------------------------------------------------------------------------
-- 2. Import Tables Géographiques
-- 2.a. import via shp2pgsql
-- 2.b. Normalisation des attributs
ALTER TABLE metropole.egd_14_sect_tir_dep34 RENAME num_secteu TO tira;
ALTER TABLE metropole.egd_14_sect_tir_dep34 DROP COLUMN object_id;
--
ALTER TABLE metropole.egd_14_zone_fine_fr RENAME num_zf TO zone_fine;
ALTER TABLE metropole.egd_14_zone_fine_fr RENAME nom_zf TO localisation_geo;
ALTER TABLE metropole.egd_14_zone_fine_fr DROP COLUMN object_id;
ALTER TABLE metropole.egd_14_zone_fine_fr RENAME num_zf_201 TO zone_fine_uid;
UPDATE metropole.egd_14_zone_fine_fr SET zone_fine_uid = translate(zone_fine_uid, ' .', '');
ALTER TABLE metropole.egd_14_zone_fine_fr ALTER COLUMN zone_fine_uid TYPE varchar(6);
-- extraction des secteur de tirage
ALTER TABLE metropole.egd_14_zone_fine_fr ADD COLUMN tira varchar(3);
UPDATE metropole.egd_14_zone_fine_fr SET tira = CASE WHEN substr(zone_fine_uid,1,3)::integer >= 200 THEN NULL ELSE substr(num_secteu,1,3) END;
-- ajout du département
ALTER TABLE metropole.egd_14_zone_fine_fr ADD COLUMN dep varchar(3);
UPDATE metropole.egd_14_zone_fine_fr SET dep =  CASE WHEN tira IS NOT NULL THEN '034'
							WHEN substr(zone_fine_uid, 1, 1) = '9' THEN '0'||substr(num_secteu, 1, 2)
							WHEN substr(zone_fine_uid, 1, 3) = '200' THEN '030'
							ELSE zone_fine
							END;
-- ajout du type d'enquête
ALTER TABLE metropole.egd_14_zone_fine_fr ADD COLUMN enquete varchar(16);
UPDATE metropole.egd_14_zone_fine_fr AS a SET enquete = replace(replace(replace(b.type_emd, 'TEL1', 'EDGT téléphone'), 'FAF', 'EDGT Face à Face'), 'TEL', 'EDVM')
					 FROM metropole.egd_14_sect_tir_dep34 AS b
					 WHERE a.tira = b.tira
					 
ALTER TABLE metropole.egd_14_zone_fine_fr ADD COLUMN tira_lib varchar(33);
UPDATE metropole.egd_14_zone_fine_fr AS a SET tira_lib = b.tira_lib
					 FROM metropole.egd_14_decoupage_interne AS b
					 WHERE a.tira = b.tira
-- Correction d'une zone fine anormale
--DROP TABLE IF EXISTS dump_zone_fine_028010;
CREATE TABLE dump_zone_fine_028010 AS 
	WITh temptab AS (
	SELECT *, (ST_Dump(the_geom)).geom as the_geom_2, st_area((ST_Dump(the_geom)).geom) as st_area_the_geom
	FROM metropole.egd_14_zone_fine_fr
	)
	SELECT gid, zone_fine_uid, localisation_geo, st_area_the_geom
	FROM temptab
	WHERE gid in (SELECT gid
			FROM temptab
			GROUP BY gid
			having count(gid)>1) AND st_area_the_geom > 1
	ORDER BY gid
	;
ALTER TABLE dump_zone_fine_028010 DROP COLUMN the_geom;
ALTER TABLE dump_zone_fine_028010 RENAME the_geom_2 TO the_geom;
DELETE FROM dump_zone_fine_028010 WHERE st_area_the_geom < 500;-- ne garde que la surface supérieure à 500 c'est à dire la plus grande
ALTER TABLE dump_zone_fine_028010 DROP COLUMN st_area_the_geom;
DELETE FROM metropole.egd_14_zone_fine_fr WHERE zone_fine_uid = '028010';
-- On réinsert les variables et on transforme la géométrie en multipolygone pour assurer la compatibilité
insert into metropole.egd_14_zone_fine_fr SELECT gid, zone_fine_uid, zone_fine, localisation_geo, num_secteu, nom_com, generateur, observatio, surface_km, shape_leng, shape_area, ST_multi(the_geom) as the_geom, tira, dep, enquete, tira_lib FROM dump_zone_fine_028010 
-- Correction du nom de deux communes dans le Gard
UPDATE metropole.egd_14_zone_fine_fr SET nom_com = 'SAINT-DIONISY' WHERE nom_com = 'SAINT-DIONIZY'
UPDATE metropole.egd_14_zone_fine_fr SET nom_com = 'PEYROLLES' WHERE nom_com = 'PEYROLES'

---------------------------------------------------------------------------------------------------
-- 3. Création des tables géographiques pour les secteurs de tirage, d30 et d10
--DROP TABLE IF EXISTS metropole.egd_14_secteur_tirage;
CREATE TABLE metropole.egd_14_secteur_tirage AS (
	SELECT tira, string_agg(distinct nom_com, '-')||' : '||tira_lib as tira_lib_long, enquete, ST_buffer(ST_buffer(ST_union(the_geom),5),-5) AS the_geom
	FROM metropole.egd_14_zone_fine_fr
	WHERE tira IS NOT NULL
	GROUP BY tira, tira_lib, enquete
	);
COMMENT ON TABLE metropole.egd_14_secteur_tirage IS 'Table générée à partir de la table metropole.egd_14_zone_fine_fr';
ALTER TABLE metropole.egd_14_secteur_tirage ADD PRIMARY KEY (tira);
CREATE INDEX egd_14_secteur_tirage_gist ON metropole.egd_14_secteur_tirage USING GIST (the_geom);
--d10
CREATE TABLE metropole.egd_14_d10 AS (
	SELECT row_number() over() as gid, b.d10, b.d10_lib, ST_buffer(ST_buffer(ST_union(the_geom),5),-5) AS the_geom
	FROM metropole.egd_14_zone_fine_fr AS a
		LEFT JOIN metropole.egd_14_decoupages_d10_d30 AS b
		ON substr(a.zone_fine_uid, 1, 3) = b.tira_prefix_zone_fine_uid
	GROUP BY b.d10, b.d10_lib
	)
--d30
CREATE TABLE metropole.egd_14_d30 AS (
	SELECT row_number() over() as gid, b.d30, b.d30_lib, ST_buffer(ST_buffer(ST_union(the_geom),5),-5) AS the_geom
	FROM metropole.egd_14_zone_fine_fr AS a
		LEFT JOIN metropole.egd_14_decoupages_d10_d30 AS b
		ON substr(a.zone_fine_uid, 1, 3) = b.tira_prefix_zone_fine_uid
	GROUP BY b.d30, b.d30_lib
	)
---------------------------------------------------------------------------------------------------
-- 4. Suppression des tables inutiles :
-- 4.a. Table géographique des secteurs géographiques : info redondante avec table des zones fines géo et qualité géométrique discutable
-- 4.b. Tables de découpage interne / externe regroupés
---------------------------------------------------------------------------------------------------
-- 5. Normalisation de la table des générateurs
ALTER TABLE metropole.egd_14_generateurs RENAME num_gene20 TO pseudo_zone_fine_generateur;
UPDATE metropole.egd_14_generateurs SET pseudo_zone_fine_generateur = translate(pseudo_zone_fine_generateur, ' ', '');
ALTER TABLE metropole.egd_14_generateurs RENAME zf_rattach TO zone_fine_uid;
UPDATE metropole.egd_14_generateurs SET zone_fine_uid = translate(zone_fine_uid, ' ', '');
ALTER TABLE metropole.egd_14_generateurs RENAME num_secteu TO tira;
UPDATE metropole.egd_14_generateurs SET tira = translate(tira, ' ', '');
ALTER TABLE metropole.egd_14_generateurs RENAME nom_com TO com_lib;
--
---------------------------------------------------------------------------------------------------
-- I. INTEGRATION EDGT FACE A FACE								---
---------------------------------------------------------------------------------------------------
-- 1. Intégration de la table ménages
-- 1.a. Création de la table dictionnaire
-- 1.a.i. Création de la structure de table dictionnaire
CREATE TABLE metropole.edgt_14_faf_menage_dessin_tab (
	position_ varchar(8),
	longueur varchar(8),
	variable varchar(8),
	variable_lib varchar(100));	
-- 1.b. Import de la table Ménage
-- 1.b.i. Création de la table
-- DROP TABLE metropole.emd04menage;
CREATE TABLE metropole.edgt_14_faf_menage (fulltxt varchar(200));
-- 1.b.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1, entête:oui, délimit: ; )
--
-- 1.b.iii. Découpage de la chaîne de caractère dans les champs d'après le dictionnaire de variable ci-dessus
SELECT metropole.set_var('metropole.edgt_14_faf_menage', 'metropole.edgt_14_faf_menage_dessin_tab');
/*-- 1.c. Post traitement : passage en année (pertinence de ces instructions à vérifier)
UPDATE metropole.edgt_14_faf_menage SET m9a = (CASE WHEN m9a < 2015 THEN 2000+m9a ELSE m9a+1900 END);
UPDATE metropole.edgt_14_faf_menage SET m9b = (CASE WHEN m9b < 2015 THEN 2000+m9b ELSE m9b+1900 END);
UPDATE metropole.edgt_14_faf_menage SET m9c = (CASE WHEN m9c < 2015 THEN 2000+m9c ELSE m9c+1900 END);
UPDATE metropole.edgt_14_faf_menage SET m9d = (CASE WHEN m9d < 2015 THEN 2000+m9d ELSE m10d+1900 END);
*/
-- 1.d. Ajout d'un identifiant unique pour chaque enregistrement et indéxation
-- 1.d.i. Modification du numéro d'échantillon du secteur de tirage étudiant doublonnant avec un numéro d'échantillon de ménage <<classique>>
-- Les numéros d'échantillon de ménages étudiants sont uniques par secteur de tirage géographique et commencent tous par 9.. (ex.: 901)
ALTER TABLE metropole.edgt_14_faf_menage ADD COLUMN ech_obsolete varchar(3);
UPDATE metropole.edgt_14_faf_menage SET ech_obsolete = ech;
UPDATE metropole.edgt_14_faf_menage AS a SET ech = b.ech_etudiant
					 FROM (SELECT tira, ech_obsolete, zfet, 900+row_number() OVER (PARTITION BY tira) as ech_etudiant FROM metropole.edgt_14_faf_menage WHERE zfet <> '      ') AS b
					 WHERE a.tira = b.tira AND a.ech_obsolete = b.ech_obsolete AND a.zfet = b.zfet
-- correction complémentaire sur la zone fine liée au secteur de tirage étudiant
ALTER TABLE metropole.edgt_14_faf_menage ADD COLUMN mp3_obsolete varchar(3);
UPDATE metropole.edgt_14_faf_menage SET mp3_obsolete = mp3;
UPDATE metropole.edgt_14_faf_menage SET mp3 = CASE WHEN mp3_obsolete ilike '9%' THEN '0'||substr(mp3_obsolete, 2,2) ELSE mp3_obsolete END;
-- 1.d.ii. création de l'identifiant unique ménage
ALTER TABLE metropole.edgt_14_faf_menage ADD COLUMN menage varchar(6);
UPDATE metropole.edgt_14_faf_menage SET menage = tira||ech;
-- 1.d.iii. Indéxation et tri suivant l'index (permet également de vérifier si un ménage est dupliqué)
CREATE UNIQUE INDEX edgt_14_faf_menage_idx ON metropole.edgt_14_faf_menage (menage);
CLUSTER metropole.edgt_14_faf_menage USING edgt_14_faf_menage_idx;
-- 1.d.iv. Création de la clé primaire
ALTER TABLE metropole.edgt_14_faf_menage ADD PRIMARY KEY (menage);

---------------------------------------------------------------------------------------------------
-- 2. Création de la table <personne>
-- 2.a. Création de la table dictionnaire
-- 2.a.i. Création de la table
CREATE TABLE metropole.edgt_14_faf_personne_dessin_tab (
	position_ varchar(8),
	longueur varchar(8),
	variable varchar(8),
	variable_lib varchar(100));
-- 2.a.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1, entête:oui, délimit: ; )
-- 2.b. Création de la table principale <personne>
-- 2.b.i. Création de la table
--DROP TABLE metropole.edgt_14_faf_personne;
CREATE TABLE metropole.edgt_14_faf_personne (fulltxt varchar(200));
-- 2.b.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1)
-- 2.b.iii. Découpage de la chaîne de caractère dans les champs d'après le dictionnaire de variable ci-dessus
SELECT metropole.set_var('metropole.edgt_14_faf_personne', 'metropole.edgt_14_faf_personne_dessin_tab', 'metropole.egd_14_normalisation');
-- 2.c. Ajout d'un identifiant unique pour chaque enregistrement et indéxation
-- 2.c.i. Modification du numéro d'échantillon erroné pour les ménages étudiants
-- correction préparatoire sur la zone fine liée au secteur de tirage étudiant
ALTER TABLE metropole.edgt_14_faf_personne ADD COLUMN pp2_obsolete varchar(3);
UPDATE metropole.edgt_14_faf_personne SET pp2_obsolete = pp2;
UPDATE metropole.edgt_14_faf_personne SET pp2 = CASE WHEN pp2_obsolete ilike '9%' THEN '0'||substr(pp2_obsolete, 2,2) ELSE pp2_obsolete END;
-- puis correction du numéro d'échantillon
ALTER TABLE metropole.edgt_14_faf_personne ADD COLUMN ech_obsolete varchar(3);
UPDATE metropole.edgt_14_faf_personne SET ech_obsolete = ech;
UPDATE metropole.edgt_14_faf_personne AS a SET ech = b.ech
					 FROM (SELECT tira, ech_obsolete, mp3_obsolete, ech FROM metropole.edgt_14_faf_menage WHERE zfet <> '      ') AS b
					 WHERE a.tira = b.tira AND a.ech_obsolete = b.ech_obsolete AND a.pp2_obsolete = b.mp3_obsolete

-- 2.c.ii. création de l'identifiant unique ménage
ALTER TABLE metropole.edgt_14_faf_personne ADD COLUMN personne varchar(8);
UPDATE metropole.edgt_14_faf_personne SET personne = tira||ech||per;
-- 2.c.iii. Indéxation et tri suivant l'index
CREATE UNIQUE INDEX edgt_14_faf_personne_idx ON metropole.edgt_14_faf_personne (personne);
CLUSTER metropole.edgt_14_faf_personne USING edgt_14_faf_personne_idx;
-- 2.c.iv. Création de la clé primaire
ALTER TABLE metropole.edgt_14_faf_personne ADD PRIMARY KEY (personne);
-- 2.d. Créer les PCS simplifiés <P11A>
ALTER TABLE metropole.edgt_14_faf_personne ADD COLUMN p11a varchar(1);
UPDATE  metropole.edgt_14_faf_personne SET p11a = CASE WHEN p11 IN ('83', '84', '85', '86', '87', '88', '89') THEN '7'
							WHEN p11 = '82' THEN '9'
							ELSE left(p11, 1) END;
---------------------------------------------------------------------------------------------------
-- 3. Création de la table <déplacement>
-- 3.a. Création de la table dictionnaire
-- 3.a.i. Création de la table
CREATE TABLE metropole.edgt_14_faf_deplacement_dessin_tab (
	position_ varchar(8),
	longueur varchar(8),
	variable varchar(8),
	variable_lib varchar(100));
-- 3.a.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1, entête:oui, délimit: ; )
-- 3.b. Création de la table principale <ménage>
-- 3.b.i. Création de la table
-- DROP TABLE IF EXISTS metropole.edgt_14_faf_deplacement;
CREATE TABLE metropole.edgt_14_faf_deplacement (fulltxt varchar(200));
-- 3.b.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1)
-- 3.b.iii. Découpage de la chaîne de caractère dans les champs d'après le dictionnaire de variable ci-dessus
SELECT metropole.set_var('metropole.edgt_14_faf_deplacement', 'metropole.edgt_14_faf_deplacement_dessin_tab');
-- 3.c. Post traitement : passage en format horaire des horodatage de début et fin de déplacement
ALTER TABLE metropole.edgt_14_faf_deplacement ADD COLUMN d4 interval;
UPDATE metropole.edgt_14_faf_deplacement SET d4 = (trim(d4a)||':'||trim(d4b))::interval;
COMMENT ON COLUMN metropole.edgt_14_faf_deplacement.d4 IS 'Heure:Minute de départ';
ALTER TABLE metropole.edgt_14_faf_deplacement ADD COLUMN d8 interval;
UPDATE metropole.edgt_14_faf_deplacement SET d8 = (trim(d8a)||':'||trim(d8b))::interval;
COMMENT ON COLUMN metropole.edgt_14_faf_deplacement.d8 IS 'Heure:Minute d arrivée';
-- 3.d. Ajout d'un identifiant unique pour chaque enregistrement et indéxation
-- 3.d.i. Modification du numéro d'échantillon erroné pour les ménages étudiants
-- correction complémentaire sur la zone fine liée au secteur de tirage étudiant
ALTER TABLE metropole.edgt_14_faf_deplacement ADD COLUMN dp2_obsolete varchar(3);
UPDATE metropole.edgt_14_faf_deplacement SET dp2_obsolete = dp2;
UPDATE metropole.edgt_14_faf_deplacement SET dp2 = CASE WHEN dp2_obsolete ilike '9%' THEN '0'||substr(dp2_obsolete, 2,2) ELSE dp2_obsolete END;
-- puis correction du numéro d'échantillon
ALTER TABLE metropole.edgt_14_faf_deplacement ADD COLUMN ech_obsolete varchar(3);
UPDATE metropole.edgt_14_faf_deplacement SET ech_obsolete = ech;
UPDATE metropole.edgt_14_faf_deplacement AS a SET ech = b.ech
					 FROM (SELECT tira, ech_obsolete, mp3_obsolete, ech FROM metropole.edgt_14_faf_menage WHERE zfet <> '      ') AS b
					 WHERE a.tira = b.tira AND a.ech_obsolete = b.ech_obsolete AND a.dp2_obsolete = b.mp3_obsolete
-- 3.d.ii. création de l'identifiant unique ménage
ALTER TABLE metropole.edgt_14_faf_deplacement ADD COLUMN deplacement varchar(10);
UPDATE metropole.edgt_14_faf_deplacement SET deplacement = tira||ech||per||ndep;
-- 3.d.iii. Indéxation et tri suivant l'index
CREATE UNIQUE INDEX edgt_14_faf_deplacement_idx ON metropole.edgt_14_faf_deplacement (deplacement);
CLUSTER metropole.edgt_14_faf_deplacement USING edgt_14_faf_deplacement_idx;
-- 3.d.iv. Création de la clé primaire
ALTER TABLE metropole.edgt_14_faf_deplacement ADD PRIMARY KEY (deplacement);
-- 3.e. Rattachement à une zone fine géographique des pôles générateurs
ALTER TABLE metropole.edgt_14_faf_deplacement ADD COLUMN d3_zone_fine_uid varchar(6);
UPDATE metropole.edgt_14_faf_deplacement SET d3_zone_fine_uid = d3;
UPDATE metropole.edgt_14_faf_deplacement AS a SET d3_zone_fine_uid = zone_fine_uid
						FROM (SELECT pseudo_zone_fine_generateur, zone_fine_uid FROM metropole.egd_14_generateurs) AS b
						WHERE a.d3 = b.pseudo_zone_fine_generateur
						;
ALTER TABLE metropole.edgt_14_faf_deplacement ADD COLUMN d7_zone_fine_uid varchar(6);
UPDATE metropole.edgt_14_faf_deplacement SET d7_zone_fine_uid = d7;
UPDATE metropole.edgt_14_faf_deplacement AS a SET d7_zone_fine_uid = zone_fine_uid
						FROM (SELECT pseudo_zone_fine_generateur, zone_fine_uid FROM metropole.egd_14_generateurs) AS b
						WHERE a.d7 = b.pseudo_zone_fine_generateur
						;
---------------------------------------------------------------------------------------------------
-- 4. Création de la table <trajet>
-- 4.a. Création de la table dictionnaire
-- 4.a.i. Création de la table
CREATE TABLE metropole.edgt_14_faf_trajet_dessin_tab (
	position_ varchar(8),
	longueur varchar(8),
	variable varchar(8),
	variable_lib varchar(100));
-- 4.a.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1, entête:oui, délimit: ; )
-- 4.b. Création de la table principale <ménage>
-- 4.b.i. Création de la table
-- DROP TABLE IF EXISTS metropole.edgt_14_faf_trajet;
CREATE TABLE metropole.edgt_14_faf_trajet (fulltxt varchar(200));
-- 4.b.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1)
-- 4.b.iii. Découpage de la chaîne de caractère dans les champs d'après le dictionnaire de variable ci-dessus
SELECT metropole.set_var('metropole.edgt_14_faf_trajet', 'metropole.edgt_14_faf_trajet_dessin_tab');
-- 4.c. Ajout d'un identifiant unique pour chaque enregistrement et indéxation
-- 4.c.i. Modification du numéro d'échantillon erroné pour les ménages étudiants
-- correction complémentaire sur la zone fine liée au secteur de tirage étudiant
ALTER TABLE metropole.edgt_14_faf_trajet ADD COLUMN tp2_obsolete varchar(3);
UPDATE metropole.edgt_14_faf_trajet SET tp2_obsolete = tp2;
UPDATE metropole.edgt_14_faf_trajet SET tp2 = CASE WHEN tp2_obsolete ilike '9%' THEN '0'||substr(tp2_obsolete, 2,2) ELSE tp2_obsolete END;
-- puis correction du numéro d'échantillon
ALTER TABLE metropole.edgt_14_faf_trajet ADD COLUMN ech_obsolete varchar(3);
UPDATE metropole.edgt_14_faf_trajet SET ech_obsolete = ech;
UPDATE metropole.edgt_14_faf_trajet AS a SET ech = b.ech
					 FROM (SELECT tira, ech_obsolete, mp3_obsolete, ech FROM metropole.edgt_14_faf_menage WHERE zfet <> '      ') AS b
					 WHERE a.tira = b.tira AND a.ech_obsolete = b.ech_obsolete AND a.tp2_obsolete = b.mp3_obsolete
-- 4.c.ii. création de l'identifiant unique ménage
ALTER TABLE metropole.edgt_14_faf_trajet ADD COLUMN trajet varchar(11);
UPDATE metropole.edgt_14_faf_trajet SET trajet = tira||ech||per||ndep||t1;
-- 4.c.iii. Indéxation et tri suivant l'index
CREATE UNIQUE INDEX edgt_14_faf_trajet_idx ON metropole.edgt_14_faf_trajet (trajet);
CLUSTER metropole.edgt_14_faf_trajet USING edgt_14_faf_trajet_idx;
-- 4.c.iv. Création de la clé primaire
ALTER TABLE metropole.edgt_14_faf_trajet ADD PRIMARY KEY (trajet);
-- 4.d. Création des hiérarchies de mode pour définir le mode principal d un déplacement
-- 4.d.i. hiérarchie urbaine : !!! commence à 2 pour respecter hiérarchie mais 1=métro absent du territoire ....
ALTER TABLE metropole.edgt_14_faf_trajet ADD COLUMN hierarchie_urbaine integer;
UPDATE metropole.edgt_14_faf_trajet SET hierarchie_urbaine =
	CASE WHEN t3 = '32' THEN 2 --32	PASSAGER TRAMWAY
		WHEN t3 = '31' THEN 3 --31 PASSAGER BUS URBAIN
		WHEN t3 = '51' THEN 4 --51 PASSAGER TRAIN (GRANDE LIGNE)
		WHEN t3 = '41' THEN 5 --41	PASSAGER  TRANSPORTS COLLECTIFS DEPARTEMENTAUX
		WHEN t3 = '42' THEN 6 --42	PASSAGER AUTRES AUTOCARS (DONT LIGNES REGIONALES, TER ROUTIERS, PERISCOLAIRE, OCCASIONNEL, LIGNES INTERNATIONALES EN CABOTAGE)
		WHEN t3 = '71' THEN 7 --71	TRANSPORT EMPLOYEUR (EXCLUSIVEMENT)
		WHEN t3 = '61' THEN 8 --61	PASSAGER TAXI
		WHEN t3 = '81' THEN 9 --81	CONDUCTEUR DE FOURGON, CAMIONNETTE, CAMION (POUR TOURNEES PROFESSIONNELLES OU DEPLACEMENTS PRIVES)
		WHEN t3 = '82' THEN 10 --82	PASSAGER DE FOURGON, CAMIONNETTE, CAMION (POUR TOURNEES PROFESSIONNELLES OU DEPLACEMENTS PRIVES)
		WHEN t3 = '95' THEN 11 --95	AUTRES MODES (TRACTEUR, ENGIN AGRICOLE, QUAD, ETC.)
		WHEN t3 = '21' THEN 12 --21	CONDUCTEUR DE VEHICULE PARTICULIER (VP)
		WHEN t3 = '22' THEN 13 --22	PASSAGER DE VEHICULE PARTICULIER (VP)
		WHEN t3 = '15' THEN 14 --15	CONDUCTEUR DE DEUX OU TROIS ROUES MOTORISES >= 50CM3
		WHEN t3 = '16' THEN 15 --16	PASSAGER DE DEUX OU TROIS ROUES MOTORISES >= 50CM3
		WHEN t3 = '13' THEN 16 --13	CONDUCTEUR DE DEUX OU TROIS ROUES MOTORISES < 50CM3
		WHEN t3 = '14' THEN 17 --14	PASSAGER DE DEUX OU TROIS ROUES MOTORISES < 50CM3
		WHEN t3 = '11' THEN 18 --11	CONDUCTEUR DE VELO
		WHEN t3 = '12' THEN 19 --12	PASSAGER DE VELO
		WHEN t3 = '93' THEN 20 --93	ROLLER, SKATE, TROTTINETTE
		WHEN t3 = '94' THEN 21 --94	FAUTEUIL ROULANT
		WHEN t3 = '39' THEN 22 --39	PASSAGER AUTRE RESEAU URBAIN 
		WHEN t3 = '91' THEN 23 --91	TRANSPORT FLUVIAL OU MARITIME
		WHEN t3 = '92' THEN 24 --92	AVION
		ELSE NULL END;
-- 4.d.i. hiérarchie interurbaine : !!! la valeur 17 correspond au passager métro absent du territoire ....
ALTER TABLE metropole.edgt_14_faf_trajet ADD COLUMN hierarchie_interurb integer;
UPDATE metropole.edgt_14_faf_trajet SET hierarchie_interurb =
	CASE 	WHEN t3 = '51' THEN 1 --51 PASSAGER TRAIN (GRANDE LIGNE)
		WHEN t3 = '41' THEN 2 --41	PASSAGER  TRANSPORTS COLLECTIFS DEPARTEMENTAUX
		WHEN t3 = '42' THEN 3 --42	PASSAGER AUTRES AUTOCARS (DONT LIGNES REGIONALES, TER ROUTIERS, PERISCOLAIRE, OCCASIONNEL, LIGNES INTERNATIONALES EN CABOTAGE)
		WHEN t3 = '71' THEN 4 --71	TRANSPORT EMPLOYEUR (EXCLUSIVEMENT)
		WHEN t3 = '61' THEN 5 --61	PASSAGER TAXI
		WHEN t3 = '81' THEN 6 --81	CONDUCTEUR DE FOURGON, CAMIONNETTE, CAMION (POUR TOURNEES PROFESSIONNELLES OU DEPLACEMENTS PRIVES)
		WHEN t3 = '21' THEN 7 --21	CONDUCTEUR DE VEHICULE PARTICULIER (VP)
		WHEN t3 = '82' THEN 8 --82	PASSAGER DE FOURGON, CAMIONNETTE, CAMION (POUR TOURNEES PROFESSIONNELLES OU DEPLACEMENTS PRIVES)
		WHEN t3 = '22' THEN 9 --22	PASSAGER DE VEHICULE PARTICULIER (VP)
		WHEN t3 = '15' THEN 10 --15	CONDUCTEUR DE DEUX OU TROIS ROUES MOTORISES >= 50CM3
		WHEN t3 = '16' THEN 11 --16	PASSAGER DE DEUX OU TROIS ROUES MOTORISES >= 50CM3
		WHEN t3 = '13' THEN 12 --13	CONDUCTEUR DE DEUX OU TROIS ROUES MOTORISES < 50CM3
		WHEN t3 = '14' THEN 13 --14	PASSAGER DE DEUX OU TROIS ROUES MOTORISES < 50CM3
		WHEN t3 = '95' THEN 14 --95	AUTRES MODES (TRACTEUR, ENGIN AGRICOLE, QUAD, ETC.)
		WHEN t3 = '91' THEN 15 --91	TRANSPORT FLUVIAL OU MARITIME
		WHEN t3 = '92' THEN 16 --92	AVION
		-- hiérarchie = 18 : PASSAGER METRO
		WHEN t3 = '32' THEN 18 --32	PASSAGER TRAMWAY
		WHEN t3 = '31' THEN 19 --31 PASSAGER BUS URBAIN
		WHEN t3 = '11' THEN 20 --11	CONDUCTEUR DE VELO
		WHEN t3 = '12' THEN 21 --12	PASSAGER DE VELO
		WHEN t3 = '93' THEN 22 --93	ROLLER, SKATE, TROTTINETTE
		WHEN t3 = '94' THEN 23 --94	FAUTEUIL ROULANT
		WHEN t3 = '39' THEN 24 --39	PASSAGER AUTRE RESEAU URBAIN 
		ELSE NULL END;
---------------------------------------------------------------------------------------------------
-- 5. Création de la table <opinion>
-- 5.a. Création de la table dictionnaire
-- 5.a.i. Création de la table
--DROP TABLE IF EXISTS metropole.edgt_14_faf_opinion_dessin_tab;
CREATE TABLE metropole.edgt_14_faf_opinion_dessin_tab (
	position_ varchar(8),
	longueur varchar(8),
	variable varchar(8),
	variable_lib varchar(140));
-- 5.a.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1, entête:oui, délimit: ; )
-- 5.b. Création de la table principale <ménage>
-- 5.b.i. Création de la table
-- DROP TABLE IF EXISTS metropole.edgt_14_faf_opinion;
CREATE TABLE metropole.edgt_14_faf_opinion (fulltxt varchar(1040));
-- ATTENTION : le délimiteur ici est un <pipe> "|"
-- 5.b.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1, entête:oui, délimit: | )
-- 5.b.iii. Découpage de la chaîne de caractère dans les champs d'après le dictionnaire de variable ci-dessus
SELECT metropole.set_var('metropole.edgt_14_faf_opinion', 'metropole.edgt_14_faf_opinion_dessin_tab');
-- 5.c. Ajout d'un identifiant unique pour chaque enregistrement et indéxation
-- 5.c.i. Modification du numéro d'échantillon erroné pour les ménages étudiants
-- correction préliminaire sur la zone fine liée au secteur de tirage étudiant
ALTER TABLE metropole.edgt_14_faf_opinion ADD COLUMN tp3_obsolete varchar(3);
UPDATE metropole.edgt_14_faf_opinion SET tp3_obsolete = tp3;
UPDATE metropole.edgt_14_faf_opinion SET tp3 = CASE WHEN tp3_obsolete ilike '9%' THEN '0'||substr(tp3_obsolete, 2,2) ELSE tp3_obsolete END;
-- puis correction du numéro d'échantillon
ALTER TABLE metropole.edgt_14_faf_opinion ADD COLUMN ech_obsolete varchar(3);
UPDATE metropole.edgt_14_faf_opinion SET ech_obsolete = ech;
UPDATE metropole.edgt_14_faf_opinion AS a SET ech = b.ech
					 FROM (SELECT tira, ech_obsolete, mp3_obsolete, ech FROM metropole.edgt_14_faf_menage WHERE zfet <> '      ') AS b
					 WHERE a.tira = b.tira AND a.ech_obsolete = b.ech_obsolete AND a.tp3_obsolete = b.mp3_obsolete
-- 5.c.ii. création de l'identifiant unique ménage
ALTER TABLE metropole.edgt_14_faf_opinion ADD COLUMN personne varchar(8);
UPDATE metropole.edgt_14_faf_opinion SET personne = tira||ech||per;
-- 5.c.iii. Indéxation et tri suivant l'index
CREATE UNIQUE INDEX edgt_14_faf_opinion_idx ON metropole.edgt_14_faf_opinion (personne);
CLUSTER metropole.edgt_14_faf_opinion USING edgt_14_faf_opinion_idx;
-- 5.c.iv. Création de la clé primaire
ALTER TABLE metropole.edgt_14_faf_opinion ADD PRIMARY KEY (personne);
---------------------------------------------------------------------------------------------------
-- 6. Ajout du mode principal urbain et interurbain à la table des déplacements
ALTER TABLE metropole.edgt_14_faf_deplacement ADD COLUMN modp_urbain int;
UPDATE metropole.edgt_14_faf_deplacement AS a SET modp_urbain = b.modp_urbain
					 FROM (	SELECT d.deplacement, min(t.hierarchie_urbaine) as modp_urbain
						FROM	metropole.edgt_14_faf_deplacement AS d
							LEFT JOIN
							metropole.edgt_14_faf_trajet AS t
							ON t.tira||t.ech||t.per||t.ndep = d.deplacement
						GROUP BY d.deplacement) AS b
					 WHERE a.deplacement = b.deplacement
ALTER TABLE metropole.edgt_14_faf_deplacement ADD COLUMN modp_interurb int;
UPDATE metropole.edgt_14_faf_deplacement AS a SET modp_interurb = b.modp_interurb
					 FROM (	SELECT d.deplacement, min(t.hierarchie_interurb) as modp_interurb
						FROM	metropole.edgt_14_faf_deplacement AS d
							LEFT JOIN
							metropole.edgt_14_faf_trajet AS t
							ON t.tira||t.ech||t.per||t.ndep = d.deplacement
						GROUP BY d.deplacement) AS b
					 WHERE a.deplacement = b.deplacement
---------------------------------------------------------------------------------------------------
--												---
-- II. INTEGRATION EDGT TELEPHONE								---
--												---
---------------------------------------------------------------------------------------------------
--
-- 1. Intégration de la table ménages
-- 1.a. Création de la table dictionnaire
-- 1.a.i. Création de la structure de table dictionnaire
CREATE TABLE metropole.edgt_14_tel_menage_dessin_tab (
	position_ varchar(8),
	longueur varchar(8),
	variable varchar(8),
	variable_lib varchar(100));
-- 1.b. Import de la table Ménage
-- 1.b.i. Création de la table
-- DROP TABLE metropole.edgt_14_tel_menage;
CREATE TABLE metropole.edgt_14_tel_menage (fulltxt varchar(200));
-- 1.b.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1, entête:oui, délimit: ; )
-- 1.b.iii. Découpage de la chaîne de caractère dans les champs d'après le dictionnaire de variable ci-dessus
SELECT metropole.set_var('metropole.edgt_14_tel_menage', 'metropole.edgt_14_tel_menage_dessin_tab');
-- 1.c. Post traitement : passage en année - ATTENTION à ne faire qu une fois
UPDATE metropole.edgt_14_tel_menage SET m10a = (CASE WHEN m10a < 5 THEN 2000+m10a ELSE m10a+1900 END);
UPDATE metropole.edgt_14_tel_menage SET m10b = (CASE WHEN m10b < 5 THEN 2000+m10b ELSE m10b+1900 END);
UPDATE metropole.edgt_14_tel_menage SET m10c = (CASE WHEN m10c < 5 THEN 2000+m10c ELSE m10c+1900 END);
UPDATE metropole.edgt_14_tel_menage SET m10d = (CASE WHEN m10d < 5 THEN 2000+m10d ELSE m10d+1900 END);
-- 1.d. Ajout d'un identifiant unique pour chaque enregistrement et indéxation
-- 1.d.i. création de l'identifiant unique ménage
ALTER TABLE metropole.edgt_14_tel_menage ADD COLUMN menage varchar(6);
UPDATE metropole.edgt_14_tel_menage SET menage = tira||ech;
-- 1.d.ii. Indéxation et tri suivant l'index
CREATE UNIQUE INDEX edgt_14_tel_menage_idx ON metropole.edgt_14_tel_menage (menage);
CLUSTER metropole.edgt_14_tel_menage USING edgt_14_tel_menage_idx;
-- 1.d.iii. Création de la clé primaire
ALTER TABLE metropole.edgt_14_tel_menage ADD PRIMARY KEY (menage);
---------------------------------------------------------------------------------------------------
-- 2. Création de la table <personne>
-- 2.a. Création de la table dictionnaire
-- 2.a.i. Création de la table
CREATE TABLE metropole.edgt_14_tel_personne_dessin_tab (
	position_ varchar(8),
	longueur varchar(8),
	variable varchar(8),
	variable_lib varchar(100));
-- 2.a.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1, entête:oui, délimit: ; )
-- 2.b. Création de la table principale <personne>
-- 2.b.i. Création de la table
--DROP TABLE metropole.edgt_14_tel_personne;
CREATE TABLE metropole.edgt_14_tel_personne (fulltxt varchar(200));
-- 2.b.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1)
-- 2.b.iii. Découpage de la chaîne de caractère dans les champs d'après le dictionnaire de variable ci-dessus
SELECT metropole.set_var('metropole.edgt_14_tel_personne', 'metropole.edgt_14_tel_personne_dessin_tab', 'metropole.egd_14_normalisation');
-- 2.c. Ajout d'un identifiant unique pour chaque enregistrement et indéxation
-- 2.c.i. création de l'identifiant unique ménage
ALTER TABLE metropole.edgt_14_tel_personne ADD COLUMN personne varchar(8);
UPDATE metropole.edgt_14_tel_personne SET personne = tira||ech||per;
-- 2.c.ii. Indéxation et tri suivant l'index
CREATE UNIQUE INDEX edgt_14_tel_personne_idx ON metropole.edgt_14_tel_personne (personne);
CLUSTER metropole.edgt_14_tel_personne USING edgt_14_tel_personne_idx;
-- 2.c.iii. Création de la clé primaire
ALTER TABLE metropole.edgt_14_tel_personne ADD PRIMARY KEY (personne);
---------------------------------------------------------------------------------------------------
-- 3. Création de la table <déplacement>
-- 3.a. Création de la table dictionnaire
-- 3.a.i. Création de la table
CREATE TABLE metropole.edgt_14_tel_deplacement_dessin_tab (
	position_ varchar(8),
	longueur varchar(8),
	variable varchar(8),
	variable_lib varchar(100));
-- 3.a.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1, entête:oui, délimit: ; )
-- 3.b. Création de la table principale <ménage>
-- 3.b.i. Création de la table
-- DROP TABLE IF EXISTS metropole.edgt_14_tel_deplacement;
CREATE TABLE metropole.edgt_14_tel_deplacement (fulltxt varchar(200));
-- 3.b.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1)
-- 3.b.iii. Découpage de la chaîne de caractère dans les champs d'après le dictionnaire de variable ci-dessus
SELECT metropole.set_var('metropole.edgt_14_tel_deplacement', 'metropole.edgt_14_tel_deplacement_dessin_tab', 'metropole.egd_14_normalisation');
-- 3.c. Post traitement : passage en format horaire des horodatage de début et fin de déplacement
ALTER TABLE metropole.edgt_14_tel_deplacement ADD COLUMN d4 interval;
UPDATE metropole.edgt_14_tel_deplacement SET d4 = (trim(d4a)||':'||trim(d4b))::interval;
COMMENT ON COLUMN metropole.edgt_14_tel_deplacement.d4 IS 'Heure:Minute de départ';
ALTER TABLE metropole.edgt_14_tel_deplacement ADD COLUMN d8 interval;
UPDATE metropole.edgt_14_tel_deplacement SET d8 = (trim(d8a)||':'||trim(d8b))::interval;
COMMENT ON COLUMN metropole.edgt_14_tel_deplacement.d8 IS 'Heure:Minute d arrivée';
-- 3.d. Ajout d'un identifiant unique pour chaque enregistrement et indéxation
-- 3.d.i. création de l'identifiant unique ménage
ALTER TABLE metropole.edgt_14_tel_deplacement ADD COLUMN deplacement varchar(10);
UPDATE metropole.edgt_14_tel_deplacement SET deplacement = tira||ech||per||ndep;
-- 3.d.ii. Indéxation et tri suivant l'index
CREATE UNIQUE INDEX edgt_14_tel_deplacement_idx ON metropole.edgt_14_tel_deplacement (deplacement);
CLUSTER metropole.edgt_14_tel_deplacement USING edgt_14_tel_deplacement_idx;
-- 3.d.iii. Création de la clé primaire
ALTER TABLE metropole.edgt_14_tel_deplacement ADD PRIMARY KEY (deplacement);
-- 3.e. Rattachement à une zone fine géographique des pôles générateurs
ALTER TABLE metropole.edgt_14_tel_deplacement ADD COLUMN d3_zone_fine_uid varchar(6);
UPDATE metropole.edgt_14_tel_deplacement SET d3_zone_fine_uid = d3;
UPDATE metropole.edgt_14_tel_deplacement AS a SET d3_zone_fine_uid = zone_fine_uid
						FROM (SELECT pseudo_zone_fine_generateur, zone_fine_uid FROM metropole.egd_14_generateurs) AS b
						WHERE a.d3 = b.pseudo_zone_fine_generateur
						;
ALTER TABLE metropole.edgt_14_tel_deplacement ADD COLUMN d7_zone_fine_uid varchar(6);
UPDATE metropole.edgt_14_tel_deplacement SET d7_zone_fine_uid = d7;
UPDATE metropole.edgt_14_tel_deplacement AS a SET d7_zone_fine_uid = zone_fine_uid
						FROM (SELECT pseudo_zone_fine_generateur, zone_fine_uid FROM metropole.egd_14_generateurs) AS b
						WHERE a.d7 = b.pseudo_zone_fine_generateur
						;
---------------------------------------------------------------------------------------------------
-- 4. Création de la table <trajet>
-- 4.a. Création de la table dictionnaire
-- 4.a.i. Création de la table
CREATE TABLE metropole.edgt_14_tel_trajet_dessin_tab (
	position_ varchar(8),
	longueur varchar(8),
	variable varchar(8),
	variable_lib varchar(100));
-- 4.a.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1, entête:oui, délimit: ; )
-- 4.b. Création de la table principale <ménage>
-- 4.b.i. Création de la table
-- DROP TABLE IF EXISTS metropole.edgt_14_tel_trajet;
CREATE TABLE metropole.edgt_14_tel_trajet (fulltxt varchar(200));
-- 4.b.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1)
-- 4.b.iii. Découpage de la chaîne de caractère dans les champs d'après le dictionnaire de variable ci-dessus
SELECT metropole.set_var('metropole.edgt_14_tel_trajet', 'metropole.edgt_14_tel_trajet_dessin_tab');
-- 4.c. Ajout d'un identifiant unique pour chaque enregistrement et indéxation
-- 4.c.i. création de l'identifiant unique ménage
ALTER TABLE metropole.edgt_14_tel_trajet ADD COLUMN trajet varchar(11);
UPDATE metropole.edgt_14_tel_trajet SET trajet = tira||ech||per||ndep||t1;
-- 4.c.ii. Indéxation et tri suivant l'index
CREATE UNIQUE INDEX edgt_14_tel_trajet_idx ON metropole.edgt_14_tel_trajet (trajet);
CLUSTER metropole.edgt_14_tel_trajet USING edgt_14_tel_trajet_idx;
-- 4.c.iii. Création de la clé primaire
ALTER TABLE metropole.edgt_14_tel_trajet ADD PRIMARY KEY (trajet);
-- 4.d. Création des hiérarchies de mode pour définir le mode principal d un déplacement
-- 4.d.i. hiérarchie urbaine : !!! commence à 2 pour respecter hiérarchie mais 1=métro absent du territoire ....
ALTER TABLE metropole.edgt_14_tel_trajet ADD COLUMN hierarchie_urbaine integer;
UPDATE metropole.edgt_14_tel_trajet SET hierarchie_urbaine =
	CASE WHEN t3 = '32' THEN 2 --32	PASSAGER TRAMWAY
		WHEN t3 = '31' THEN 3 --31 PASSAGER BUS URBAIN
		WHEN t3 = '51' THEN 4 --51 PASSAGER TRAIN (GRANDE LIGNE)
		WHEN t3 = '41' THEN 5 --41	PASSAGER  TRANSPORTS COLLECTIFS DEPARTEMENTAUX
		WHEN t3 = '42' THEN 6 --42	PASSAGER AUTRES AUTOCARS (DONT LIGNES REGIONALES, TER ROUTIERS, PERISCOLAIRE, OCCASIONNEL, LIGNES INTERNATIONALES EN CABOTAGE)
		WHEN t3 = '71' THEN 7 --71	TRANSPORT EMPLOYEUR (EXCLUSIVEMENT)
		WHEN t3 = '61' THEN 8 --61	PASSAGER TAXI
		WHEN t3 = '81' THEN 9 --81	CONDUCTEUR DE FOURGON, CAMIONNETTE, CAMION (POUR TOURNEES PROFESSIONNELLES OU DEPLACEMENTS PRIVES)
		WHEN t3 = '82' THEN 10 --82	PASSAGER DE FOURGON, CAMIONNETTE, CAMION (POUR TOURNEES PROFESSIONNELLES OU DEPLACEMENTS PRIVES)
		WHEN t3 = '95' THEN 11 --95	AUTRES MODES (TRACTEUR, ENGIN AGRICOLE, QUAD, ETC.)
		WHEN t3 = '21' THEN 12 --21	CONDUCTEUR DE VEHICULE PARTICULIER (VP)
		WHEN t3 = '22' THEN 13 --22	PASSAGER DE VEHICULE PARTICULIER (VP)
		WHEN t3 = '15' THEN 14 --15	CONDUCTEUR DE DEUX OU TROIS ROUES MOTORISES >= 50CM3
		WHEN t3 = '16' THEN 15 --16	PASSAGER DE DEUX OU TROIS ROUES MOTORISES >= 50CM3
		WHEN t3 = '13' THEN 16 --13	CONDUCTEUR DE DEUX OU TROIS ROUES MOTORISES < 50CM3
		WHEN t3 = '14' THEN 17 --14	PASSAGER DE DEUX OU TROIS ROUES MOTORISES < 50CM3
		WHEN t3 = '11' THEN 18 --11	CONDUCTEUR DE VELO
		WHEN t3 = '12' THEN 19 --12	PASSAGER DE VELO
		WHEN t3 = '93' THEN 20 --93	ROLLER, SKATE, TROTTINETTE
		WHEN t3 = '94' THEN 21 --94	FAUTEUIL ROULANT
		WHEN t3 = '39' THEN 22 --39	PASSAGER AUTRE RESEAU URBAIN 
		WHEN t3 = '91' THEN 23 --91	TRANSPORT FLUVIAL OU MARITIME
		WHEN t3 = '92' THEN 24 --92	AVION
		ELSE NULL END;
-- 4.d.i. hiérarchie interurbaine : !!! la valeur 17 correspond au passager métro absent du territoire ....
ALTER TABLE metropole.edgt_14_tel_trajet ADD COLUMN hierarchie_interurb integer;
UPDATE metropole.edgt_14_tel_trajet SET hierarchie_interurb =
	CASE 	WHEN t3 = '51' THEN 1 --51 PASSAGER TRAIN (GRANDE LIGNE)
		WHEN t3 = '41' THEN 2 --41	PASSAGER  TRANSPORTS COLLECTIFS DEPARTEMENTAUX
		WHEN t3 = '42' THEN 3 --42	PASSAGER AUTRES AUTOCARS (DONT LIGNES REGIONALES, TER ROUTIERS, PERISCOLAIRE, OCCASIONNEL, LIGNES INTERNATIONALES EN CABOTAGE)
		WHEN t3 = '71' THEN 4 --71	TRANSPORT EMPLOYEUR (EXCLUSIVEMENT)
		WHEN t3 = '61' THEN 5 --61	PASSAGER TAXI
		WHEN t3 = '81' THEN 6 --81	CONDUCTEUR DE FOURGON, CAMIONNETTE, CAMION (POUR TOURNEES PROFESSIONNELLES OU DEPLACEMENTS PRIVES)
		WHEN t3 = '21' THEN 7 --21	CONDUCTEUR DE VEHICULE PARTICULIER (VP)
		WHEN t3 = '82' THEN 8 --82	PASSAGER DE FOURGON, CAMIONNETTE, CAMION (POUR TOURNEES PROFESSIONNELLES OU DEPLACEMENTS PRIVES)
		WHEN t3 = '22' THEN 9 --22	PASSAGER DE VEHICULE PARTICULIER (VP)
		WHEN t3 = '15' THEN 10 --15	CONDUCTEUR DE DEUX OU TROIS ROUES MOTORISES >= 50CM3
		WHEN t3 = '16' THEN 11 --16	PASSAGER DE DEUX OU TROIS ROUES MOTORISES >= 50CM3
		WHEN t3 = '13' THEN 12 --13	CONDUCTEUR DE DEUX OU TROIS ROUES MOTORISES < 50CM3
		WHEN t3 = '14' THEN 13 --14	PASSAGER DE DEUX OU TROIS ROUES MOTORISES < 50CM3
		WHEN t3 = '95' THEN 14 --95	AUTRES MODES (TRACTEUR, ENGIN AGRICOLE, QUAD, ETC.)
		WHEN t3 = '91' THEN 15 --91	TRANSPORT FLUVIAL OU MARITIME
		WHEN t3 = '92' THEN 16 --92	AVION
		-- hiérarchie = 17 : PASSAGER METRO
		WHEN t3 = '32' THEN 18 --32	PASSAGER TRAMWAY
		WHEN t3 = '31' THEN 19 --31 PASSAGER BUS URBAIN
		WHEN t3 = '11' THEN 20 --11	CONDUCTEUR DE VELO
		WHEN t3 = '12' THEN 21 --12	PASSAGER DE VELO
		WHEN t3 = '93' THEN 22 --93	ROLLER, SKATE, TROTTINETTE
		WHEN t3 = '94' THEN 23 --94	FAUTEUIL ROULANT
		WHEN t3 = '39' THEN 24 --39	PASSAGER AUTRE RESEAU URBAIN 
		ELSE NULL END;
-- 5. Ajout du mode principal urbain et interurbain à la table des déplacements
ALTER TABLE metropole.edgt_14_tel_deplacement ADD COLUMN modp_urbain int;
UPDATE metropole.edgt_14_tel_deplacement AS a SET modp_urbain = b.modp_urbain
					 FROM (	SELECT d.deplacement, min(t.hierarchie_urbaine) as modp_urbain
						FROM	metropole.edgt_14_tel_deplacement AS d
							LEFT JOIN
							metropole.edgt_14_tel_trajet AS t
							ON t.tira||t.ech||t.per||t.ndep = d.deplacement
						GROUP BY d.deplacement) AS b
					 WHERE a.deplacement = b.deplacement
ALTER TABLE metropole.edgt_14_tel_deplacement ADD COLUMN modp_interurb int;
UPDATE metropole.edgt_14_tel_deplacement AS a SET modp_interurb = b.modp_interurb
					 FROM (	SELECT d.deplacement, min(t.hierarchie_interurb) as modp_interurb
						FROM	metropole.edgt_14_tel_deplacement AS d
							LEFT JOIN
							metropole.edgt_14_tel_trajet AS t
							ON t.tira||t.ech||t.per||t.ndep = d.deplacement
						GROUP BY d.deplacement) AS b
					 WHERE a.deplacement = b.deplacement
--
---------------------------------------------------------------------------------------------------
--												---
-- III. INTEGRATION EDVM									---
--												---
---------------------------------------------------------------------------------------------------
-- 1. Intégration de la table ménages
-- 1.a. Création de la table dictionnaire
-- 1.a.i. Création de la structure de table dictionnaire
CREATE TABLE metropole.edvm_14_menage_dessin_tab (
	position_ varchar(8),
	longueur varchar(8),
	variable varchar(8),
	variable_lib varchar(100));
-- 1.b. Import de la table Ménage
-- 1.b.i. Création de la table
-- DROP TABLE metropole.edvm_14_menage;
CREATE TABLE metropole.edvm_14_menage (fulltxt varchar(200));
-- 1.b.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1, entête:oui, délimit: ; )
-- 1.b.iii. Découpage de la chaîne de caractère dans les champs d'après le dictionnaire de variable ci-dessus
SELECT metropole.set_var('metropole.edvm_14_menage', 'metropole.edvm_14_menage_dessin_tab', 'metropole.egd_14_normalisation');
-- 1.c. Post traitement : passage en année - ATTENTION à ne faire qu une fois
-- !!! ATTENTION !!! On ne considère que les deux premiers véhicules
UPDATE metropole.edvm_14_menage SET m10a = (CASE WHEN m10a < 5 THEN 2000+m10a ELSE m10a+1900 END);
UPDATE metropole.edvm_14_menage SET m10b = (CASE WHEN m10b < 5 THEN 2000+m10b ELSE m10b+1900 END);
-- 1.d. Ajout d'un identifiant unique pour chaque enregistrement et indéxation
-- 1.d.i. création de l'identifiant unique ménage
ALTER TABLE metropole.edvm_14_menage ADD COLUMN menage varchar(7);
UPDATE metropole.edvm_14_menage SET menage = tira||ech;
-- 1.d.ii. Indéxation et tri suivant l'index
CREATE UNIQUE INDEX edvm_14_menage_idx ON metropole.edvm_14_menage (menage);
CLUSTER metropole.edvm_14_menage USING edvm_14_menage_idx;
-- 1.d.iii. Création de la clé primaire
ALTER TABLE metropole.edvm_14_menage ADD PRIMARY KEY (menage);
---------------------------------------------------------------------------------------------------
-- 2. Création de la table <personne>
-- 2.a. Création de la table dictionnaire
-- 2.a.i. Création de la table
CREATE TABLE metropole.edvm_14_personne_dessin_tab (
	position_ varchar(8),
	longueur varchar(8),
	variable varchar(8),
	variable_lib varchar(100));
-- 2.a.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1, entête:oui, délimit: ; )
-- 2.b. Création de la table principale <personne>
-- 2.b.i. Création de la table
--DROP TABLE metropole.edvm_14_personne;
CREATE TABLE metropole.edvm_14_personne (fulltxt varchar(200));
-- 2.b.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1)
-- 2.b.iii. Découpage de la chaîne de caractère dans les champs d'après le dictionnaire de variable ci-dessus
SELECT metropole.set_var('metropole.edvm_14_personne', 'metropole.edvm_14_personne_dessin_tab', 'metropole.egd_14_normalisation');
-- 2.c. Ajout d'un identifiant unique pour chaque enregistrement et indéxation
-- 2.c.i. création de l'identifiant unique ménage
ALTER TABLE metropole.edvm_14_personne ADD COLUMN personne varchar(9);
UPDATE metropole.edvm_14_personne SET personne = tira||ech||per;
-- 2.c.ii. Indéxation et tri suivant l'index
CREATE UNIQUE INDEX edvm_14_personne_idx ON metropole.edvm_14_personne (personne);
CLUSTER metropole.edvm_14_personne USING edvm_14_personne_idx;
-- 2.c.iii. Création de la clé primaire
ALTER TABLE metropole.edvm_14_personne ADD PRIMARY KEY (personne);
---------------------------------------------------------------------------------------------------
-- 3. Création de la table <déplacement>
-- 3.a. Création de la table dictionnaire
-- 3.a.i. Création de la table
-- !!!! ATTENTION: variable codée sur 32 caractères !!!
CREATE TABLE metropole.edvm_14_deplacement_dessin_tab (
	position_ varchar(8),
	longueur varchar(8),
	variable varchar(32),
	variable_lib varchar(100));
-- 3.a.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1, entête:oui, délimit: ; )
-- 3.b. Création de la table principale <ménage>
-- 3.b.i. Création de la table
-- DROP TABLE IF EXISTS metropole.edvm_14_deplacement;
CREATE TABLE metropole.edvm_14_deplacement (fulltxt varchar(200));
-- 3.b.ii. Import tab : PGAdmin>table>clic droit>import (Format:CSV, codage : LATIN-1)
-- 3.b.iii. Découpage de la chaîne de caractère dans les champs d'après le dictionnaire de variable ci-dessus
SELECT metropole.set_var('metropole.edvm_14_deplacement', 'metropole.edvm_14_deplacement_dessin_tab', 'metropole.egd_14_normalisation');
-- 3.c. Post traitement : passage en format horaire des horodatage de début et fin de déplacement
ALTER TABLE metropole.edvm_14_deplacement ADD COLUMN d4_temp interval;
UPDATE metropole.edvm_14_deplacement SET d4_temp = (substr(d4,1,2)||':'||substr(d4,3,2))::interval;
ALTER TABLE metropole.edvm_14_deplacement RENAME COLUMN d4 to d4txt;
ALTER TABLE metropole.edvm_14_deplacement RENAME COLUMN d4_temp to d4;
COMMENT ON COLUMN metropole.edvm_14_deplacement.d4 IS 'Heure:Minute de départ (peut être supérieur à 24 heures)';
ALTER TABLE metropole.edvm_14_deplacement ADD COLUMN d8_temp interval;
UPDATE metropole.edvm_14_deplacement SET d8_temp = (substr(d8,1,2)||':'||substr(d8,3,2))::interval;
ALTER TABLE metropole.edvm_14_deplacement RENAME COLUMN d8 to d8txt;
ALTER TABLE metropole.edvm_14_deplacement RENAME COLUMN d8_temp to d8;
COMMENT ON COLUMN metropole.edvm_14_deplacement.d8 IS 'Heure:Minute d arrivée (peut être supérieur à 24 heures)';
-- 3.d. Ajout d'un identifiant unique pour chaque enregistrement et indéxation
-- 3.d.i. création de l'identifiant unique ménage
ALTER TABLE metropole.edvm_14_deplacement ADD COLUMN deplacement varchar(11);
UPDATE metropole.edvm_14_deplacement SET deplacement = tira||ech||per||ndep;
-- 3.d.ii. Indéxation et tri suivant l'index
CREATE UNIQUE INDEX edvm_14_deplacement_idx ON metropole.edvm_14_deplacement (deplacement);
CLUSTER metropole.edvm_14_deplacement USING edvm_14_deplacement_idx;
-- 3.d.iii. Création de la clé primaire
ALTER TABLE metropole.edvm_14_deplacement ADD PRIMARY KEY (deplacement);
--
-- 3.e. Rattachement à une zone fine géographique des pôles générateurs
ALTER TABLE metropole.edvm_14_deplacement ADD COLUMN d3_zone_fine_uid varchar(6);
UPDATE metropole.edvm_14_deplacement SET d3_zone_fine_uid = d3;
UPDATE metropole.edvm_14_deplacement AS a SET d3_zone_fine_uid = zone_fine_uid
						FROM (SELECT pseudo_zone_fine_generateur, zone_fine_uid FROM metropole.egd_14_generateurs) AS b
						WHERE a.d3 = b.pseudo_zone_fine_generateur
						;
ALTER TABLE metropole.edvm_14_deplacement ADD COLUMN d7_zone_fine_uid varchar(6);
UPDATE metropole.edvm_14_deplacement SET d7_zone_fine_uid = d7;
UPDATE metropole.edvm_14_deplacement AS a SET d7_zone_fine_uid = zone_fine_uid
						FROM (SELECT pseudo_zone_fine_generateur, zone_fine_uid FROM metropole.egd_14_generateurs) AS b
						WHERE a.d7 = b.pseudo_zone_fine_generateur
						;
-- 3.f. Conversion du mode principal
ALTER TABLE metropole.edvm_14_deplacement ADD COLUMN modp_urbain integer;
UPDATE metropole.edvm_14_deplacement SET modp_urbain = 
	CASE WHEN modp = '32' THEN 2 --32	PASSAGER TRAMWAY
		WHEN modp = '31' THEN 3 --31 PASSAGER BUS URBAIN
		WHEN modp = '51' THEN 4 --51 PASSAGER TRAIN (GRANDE LIGNE)
		WHEN modp = '41' THEN 5 --41	PASSAGER  TRANSPORTS COLLECTIFS DEPARTEMENTAUX
		WHEN modp = '42' THEN 6 --42	PASSAGER AUTRES AUTOCARS (DONT LIGNES REGIONALES, TER ROUTIERS, PERISCOLAIRE, OCCASIONNEL, LIGNES INTERNATIONALES EN CABOTAGE)
		WHEN modp = '71' THEN 7 --71	TRANSPORT EMPLOYEUR (EXCLUSIVEMENT)
		WHEN modp = '61' THEN 8 --61	PASSAGER TAXI
		WHEN modp = '81' THEN 9 --81	CONDUCTEUR DE FOURGON, CAMIONNETTE, CAMION (POUR TOURNEES PROFESSIONNELLES OU DEPLACEMENTS PRIVES)
		WHEN modp = '82' THEN 10 --82	PASSAGER DE FOURGON, CAMIONNETTE, CAMION (POUR TOURNEES PROFESSIONNELLES OU DEPLACEMENTS PRIVES)
		WHEN modp = '95' THEN 11 --95	AUTRES MODES (TRACTEUR, ENGIN AGRICOLE, QUAD, ETC.)
		WHEN modp = '21' THEN 12 --21	CONDUCTEUR DE VEHICULE PARTICULIER (VP)
		WHEN modp = '22' THEN 13 --22	PASSAGER DE VEHICULE PARTICULIER (VP)
		WHEN modp = '15' THEN 14 --15	CONDUCTEUR DE DEUX OU TROIS ROUES MOTORISES >= 50CM3
		WHEN modp = '16' THEN 15 --16	PASSAGER DE DEUX OU TROIS ROUES MOTORISES >= 50CM3
		WHEN modp = '13' THEN 16 --13	CONDUCTEUR DE DEUX OU TROIS ROUES MOTORISES < 50CM3
		WHEN modp = '14' THEN 17 --14	PASSAGER DE DEUX OU TROIS ROUES MOTORISES < 50CM3
		WHEN modp = '11' THEN 18 --11	CONDUCTEUR DE VELO
		WHEN modp = '12' THEN 19 --12	PASSAGER DE VELO
		WHEN modp = '93' THEN 20 --93	ROLLER, SKATE, TROTTINETTE
		WHEN modp = '94' THEN 21 --94	FAUTEUIL ROULANT
		WHEN modp = '39' THEN 22 --39	PASSAGER AUTRE RESEAU URBAIN 
		WHEN modp = '91' THEN 23 --91	TRANSPORT FLUVIAL OU MARITIME
		WHEN modp = '92' THEN 24 --92	AVION
		ELSE NULL END;

ALTER TABLE metropole.edvm_14_deplacement ADD COLUMN modp_interurb integer;
UPDATE metropole.edvm_14_deplacement SET modp_interurb =
	CASE 	WHEN moip = '51' THEN 1 --51 PASSAGER TRAIN (GRANDE LIGNE)
		WHEN moip = '41' THEN 2 --41	PASSAGER  TRANSPORTS COLLECTIFS DEPARTEMENTAUX
		WHEN moip = '42' THEN 3 --42	PASSAGER AUTRES AUTOCARS (DONT LIGNES REGIONALES, TER ROUTIERS, PERISCOLAIRE, OCCASIONNEL, LIGNES INTERNATIONALES EN CABOTAGE)
		WHEN moip = '71' THEN 4 --71	TRANSPORT EMPLOYEUR (EXCLUSIVEMENT)
		WHEN moip = '61' THEN 5 --61	PASSAGER TAXI
		WHEN moip = '81' THEN 6 --81	CONDUCTEUR DE FOURGON, CAMIONNETTE, CAMION (POUR TOURNEES PROFESSIONNELLES OU DEPLACEMENTS PRIVES)
		WHEN moip = '21' THEN 7 --21	CONDUCTEUR DE VEHICULE PARTICULIER (VP)
		WHEN moip = '82' THEN 8 --82	PASSAGER DE FOURGON, CAMIONNETTE, CAMION (POUR TOURNEES PROFESSIONNELLES OU DEPLACEMENTS PRIVES)
		WHEN moip = '22' THEN 9 --22	PASSAGER DE VEHICULE PARTICULIER (VP)
		WHEN moip = '15' THEN 10 --15	CONDUCTEUR DE DEUX OU TROIS ROUES MOTORISES >= 50CM3
		WHEN moip = '16' THEN 11 --16	PASSAGER DE DEUX OU TROIS ROUES MOTORISES >= 50CM3
		WHEN moip = '13' THEN 12 --13	CONDUCTEUR DE DEUX OU TROIS ROUES MOTORISES < 50CM3
		WHEN moip = '14' THEN 13 --14	PASSAGER DE DEUX OU TROIS ROUES MOTORISES < 50CM3
		WHEN moip = '95' THEN 14 --95	AUTRES MODES (TRACTEUR, ENGIN AGRICOLE, QUAD, ETC.)
		WHEN moip = '91' THEN 15 --91	TRANSPORT FLUVIAL OU MARITIME
		WHEN moip = '92' THEN 16 --92	AVION
		-- hiérarchie = 17 : PASSAGER METRO
		WHEN moip = '32' THEN 18 --32	PASSAGER TRAMWAY
		WHEN moip = '31' THEN 19 --31 PASSAGER BUS URBAIN
		WHEN moip = '11' THEN 20 --11	CONDUCTEUR DE VELO
		WHEN moip = '12' THEN 21 --12	PASSAGER DE VELO
		WHEN moip = '93' THEN 22 --93	ROLLER, SKATE, TROTTINETTE
		WHEN moip = '94' THEN 23 --94	FAUTEUIL ROULANT
		WHEN moip = '39' THEN 24 --39	PASSAGER AUTRE RESEAU URBAIN 
		ELSE NULL END;
----------------------------------------------------------------------------------------------------------
-- POST TRAITEMENT : compilation dictionnaires et dessins de variables
----------------------------------------------------------------------------------------------------------
-- 1. Ajout des dessins de variables
--ALTER TABLE metropole._dessin_variable ALTER COLUMN variable TYPE varchar(32)
--  FROM metropole._dessin_variable WHERE table_name ilike '%_14_%'
SELECT metropole.compil_dessin('metropole')
--
----------------------------------------------------------------------------------------------------------
-- IV. CONTROLE : Vérification des grands volumes
----------------------------------------------------------------------------------------------------------
-- 1. Nombre de ménages
-- 1.a. D'après EMD 2014 (cible : 347891 EDGT + 129164 EDVM)
DROP VIEW IF EXISTS metropole.egd_14_menage;
CREATE OR REPLACE VIEW metropole.egd_14_menage AS (
	SELECT tira, ech, menage, coe0
	FROM metropole.edgt_14_faf_menage
	UNION
	SELECT tira, ech, menage, coe0
	FROM metropole.edgt_14_tel_menage
	UNION
	SELECT tira, ech, menage, coem as coe0
	FROM metropole.edvm_14_menage
	);
-- 1.b. Comparer les ménages annoncés au total redressé de la base de données
SELECT 1 as total_perimetre, sum(coe0) as menages, 347891+129164 as menages_exploit_std_cm1
FROM metropole.egd_14_menage
-- 2. Nombre de personnes (coefficient appliqué aussi aux non enquêtés et moins de 5 ans)
-- 697393 personnes EDGT FàF, 48364 personnes EDGT Tél, 283367 personnes EDVM
-- dont inf 5 ans : 38627 EDGT FàF, 3000 EDGT Tél, 15621 EDVM
DROP VIEW IF EXISTS metropole.egd_14_personne;
CREATE OR REPLACE VIEW metropole.egd_14_personne AS (
	SELECT tira, ech, per, personne, p2, p4, p9, p11a, coep
	FROM metropole.edgt_14_faf_personne
	UNION
	SELECT tira, ech, per, personne, p2, p4, p9, p11a,coe1 as coep
	FROM metropole.edgt_14_tel_personne
	UNION
	SELECT tira, ech, per, personne, p2, p4, p9, p11a, coe1 as coep
	FROM metropole.edvm_14_personne
	)
-- 2.b. Comparer le nombre de personnes annoncées au total redressé de la base de données
SELECT 1 as total_perimetre, sum(coep) as personnes, 697393+48364+283367 as personnes_exploit_std_cp1
FROM metropole.egd_14_personne
---------------------------------------------------------------------------------------------------
-- 3. Nombre de déplacements
--  (cible : 2626562 FAF, 146541 Tél, 1030036 EDVM - source : cd1)
-- Création de vue globale
DROP VIEW IF EXISTS metropole.egd_14_deplacement;
CREATE OR REPLACE VIEW metropole.egd_14_deplacement AS (
	SELECT tira, ech, per, deplacement, d3_zone_fine_uid, d7_zone_fine_uid, d2, d4, d5, d8, d10, modp_urbain, 'faf'::text as enquete
	FROM metropole.edgt_14_faf_deplacement
	UNION
	SELECT tira, ech, per, deplacement, d3_zone_fine_uid, d7_zone_fine_uid, d2, d4, d5, d8, d10, modp_urbain, 'tel'::text as enquete
	FROM metropole.edgt_14_tel_deplacement
	UNION
	SELECT tira, ech, per, deplacement, d3_zone_fine_uid, d7_zone_fine_uid, d2, d4, d5, d8, d10, modp_urbain, 'edvm'::text as enquete
	FROM metropole.edvm_14_deplacement
	)
DROP VIEW IF EXISTS metropole.egd_14_deplacement_personne;
CREATE OR REPLACE VIEW metropole.egd_14_deplacement_personne AS (
	SELECT a.*, b.*
	FROM	metropole.egd_14_deplacement AS a
		LEFT JOIN
		(SELECT personne, coep, p2, p4, p9, p11a
		FROM metropole.edgt_14_faf_personne
		UNION
		-- ATTENTION : par rapport au calcul de volume de personnes, pour le volume de déplacements, on prend coep et non coe1
		SELECT personne, coep, p2, p4, p9, p11a
		FROM metropole.edgt_14_tel_personne
		UNION
		SELECT personne, coep, p2, p4, p9, p11a
		FROM metropole.edvm_14_personne) AS b
		ON a.tira||a.ech||a.per = b.personne
	WHERE enquete <> 'edvm' AND (a.d3_zone_fine_uid IN (SELECT zone_fine_uid FROM metropole.egd_14_zone_fine_fr WHERE enquete <> 'EDVM' AND enquete IS NOT NULL)
								OR a.d7_zone_fine_uid IN (SELECT zone_fine_uid FROM metropole.egd_14_zone_fine_fr WHERE enquete <> 'EDVM' AND enquete IS NOT NULL))
				OR enquete = 'edvm' AND (a.d3_zone_fine_uid IN (SELECT zone_fine_uid FROM metropole.egd_14_zone_fine_fr WHERE enquete = 'EDVM' AND enquete IS NOT NULL)
								OR a.d7_zone_fine_uid IN (SELECT zone_fine_uid FROM metropole.egd_14_zone_fine_fr WHERE enquete = 'EDVM' AND enquete IS NOT NULL))
	)
-- Requête de vérification avec des modes de transport simplifié <<Mod9>>
WITH temp_dep AS (
	SELECT *
		, CASE WHEN modp_urbain=12 THEN 'MOD9-1-VPC'
			WHEN modp_urbain=13 THEN 'MOD9-2-VPP'
			WHEN modp_urbain<=3 THEN 'MOD9-3-TCU'
			WHEN modp_urbain between 4 AND 7 THEN 'MOD9-4-AUTRE TC'
			WHEN modp_urbain between 18 AND 19 THEN 'MOD9-5-VELO'
			--WHEN modp_urbain between 16 AND 17 THEN 'MOD9-6-2RM inf50cm3'
			--WHEN modp_urbain between 14 AND 15 THEN 'MOD9-7-2RM sup50cm3'
			WHEN modp_urbain between 14 AND 17 THEN 'MOD9-7-2RM'
			WHEN d10=0 THEN 'MOD9-8-Marche à pied'
			WHEN modp_urbain between 8 AND 11 OR modp_urbain>=20 THEN 'MOD9-9-Autre'
			ELSE NULL END AS mod9
	FROM	metropole.egd_14_deplacement_personne
	)
-- Total du nombre de déplacements redressés par type d'enquête
SELECT enquete, sum(coep) AS deplacements
FROM temp_dep
GROUP BY enquete
ORDER BY enquete

-- Requête finale de vérification par secteur de tirage et mode
CREATE TABLE egd_14_dep_vp_tira AS (
	WITH temp_dep AS (
		SELECT *
			, CASE WHEN modp_urbain=12 THEN 'MOD9-1-VPC'
				WHEN modp_urbain=13 THEN 'MOD9-2-VPP'
				WHEN modp_urbain<=3 THEN 'MOD9-3-TCU'
				WHEN modp_urbain between 4 AND 7 THEN 'MOD9-4-AUTRE TC'
				WHEN modp_urbain between 18 AND 19 THEN 'MOD9-5-VELO'
				--WHEN modp_urbain between 16 AND 17 THEN 'MOD9-6-2RM inf50cm3'
				--WHEN modp_urbain between 14 AND 15 THEN 'MOD9-7-2RM sup50cm3'
				WHEN modp_urbain between 14 AND 17 THEN 'MOD9-7-2RM'
				WHEN d10=0 THEN 'MOD9-8-Marche à pied'
				WHEN modp_urbain between 8 AND 11 OR modp_urbain>=20 THEN 'MOD9-9-Autre'
				ELSE NULL END AS mod9
		FROM	metropole.egd_14_deplacement_personne
		)
	, vp_par_tira AS (
		SELECT tira, sum(coep) AS deplacements_vp
		FROM temp_dep
		WHERE mod9 IN ('MOD9-1-VPC', 'MOD9-2-VPP')
		GROUP BY tira
		ORDER BY tira
		)
	, dep_par_tira AS (
		SELECT tira, sum(coep) AS deplacements
		FROM temp_dep
		GROUP BY tira
		ORDER BY tira
		)
	, per_par_tira AS (
		SELECT tira, sum(coep) as personnes
		FROM metropole.egd_14_personne
		GROUP BY tira
		)
	SELECT a.tira, a.deplacements_vp, b.deplacements, c.personnes, a.deplacements_vp/c.personnes AS dep_vp_par_per
	FROM vp_par_tira AS a
		, dep_par_tira AS b
		, per_par_tira AS c
	WHERE a.tira = b.tira AND a.tira = c.tira
	)